<?php
/**
 * Plugin Name: AI Writer API Whitelist
 * Plugin URI: https://github.com/alankongx/ai-writer-toolbox-zeabur
 * Description: 允许 AI Writer 工具箱通过 WordPress REST API 上传图片和发布文章，绕过 SiteGround 等安全插件的拦截。支持 Sideload 模式（WordPress 服务器自行下载图片）+ 自动 Resize 到 600x400。
 * Version: 2.1.0
 * Author: AI Writer
 * Author URI: https://github.com/alankongx
 * License: MIT
 * Text Domain: ai-writer-api-whitelist
 */

if (!defined('ABSPATH')) {
    exit; // 禁止直接访问
}

class AI_Writer_API_Whitelist {
    
    // 插件版本
    const VERSION = '2.1.0';
    
    // AI Writer 自定义图片尺寸
    const AIWRITER_IMAGE_WIDTH = 600;
    const AIWRITER_IMAGE_HEIGHT = 400;
    
    // AI Writer 服务器 IP 白名单
    private $whitelisted_ips = array(
        '43.166.139.249',  // Zeabur 主服务器
        // 可以添加更多 IP
    );
    
    public function __construct() {
        // 在 WordPress 初始化早期运行，优先于安全插件
        add_action('plugins_loaded', array($this, 'maybe_bypass_security'), 1);
        
        // 添加自定义 REST API 认证过滤器
        add_filter('rest_authentication_errors', array($this, 'allow_ai_writer_requests'), 1);
        
        // 禁用特定请求的安全检查
        add_action('init', array($this, 'disable_security_for_api'), 1);
        
        // 注册自定义 REST API 端点
        add_action('rest_api_init', array($this, 'register_rest_routes'));
        
        // 添加管理页面
        add_action('admin_menu', array($this, 'add_admin_menu'));
        
        // 注册设置
        add_action('admin_init', array($this, 'register_settings'));
        
        // 注册自定义图片尺寸
        add_action('after_setup_theme', array($this, 'register_custom_image_sizes'));
    }
    
    /**
     * 注册 AI Writer 自定义图片尺寸
     */
    public function register_custom_image_sizes() {
        // 600x400 用于文章配图（3:2 比例，居中裁剪）
        add_image_size('aiwriter-article', self::AIWRITER_IMAGE_WIDTH, self::AIWRITER_IMAGE_HEIGHT, true);
        // 600x600 用于封面图（1:1 比例，居中裁剪）
        add_image_size('aiwriter-cover', 600, 600, true);
    }
    
    /**
     * 注册自定义 REST API 端点
     */
    public function register_rest_routes() {
        // Ping 端点：用于检测插件是否安装
        register_rest_route('ai-writer/v1', '/ping', array(
            'methods' => 'GET',
            'callback' => array($this, 'handle_ping'),
            'permission_callback' => '__return_true', // 公开端点
        ));
        
        // Sideload 端点：接收图片 URL，WordPress 自行下载并存入媒体库
        register_rest_route('ai-writer/v1', '/sideload', array(
            'methods' => 'POST',
            'callback' => array($this, 'handle_sideload'),
            'permission_callback' => array($this, 'check_upload_permission'),
            'args' => array(
                'image_url' => array(
                    'required' => true,
                    'type' => 'string',
                    'description' => '要下载的图片 URL',
                    'validate_callback' => function($value) {
                        return filter_var($value, FILTER_VALIDATE_URL) !== false;
                    },
                ),
                'filename' => array(
                    'required' => false,
                    'type' => 'string',
                    'description' => '保存的文件名',
                    'default' => '',
                ),
                'alt_text' => array(
                    'required' => false,
                    'type' => 'string',
                    'description' => '图片的 alt 文本',
                    'default' => '',
                ),
                'title' => array(
                    'required' => false,
                    'type' => 'string',
                    'description' => '图片标题',
                    'default' => '',
                ),
                'resize' => array(
                    'required' => false,
                    'type' => 'string',
                    'description' => '调整尺寸: article(600x400), cover(600x600), 或 WxH 格式',
                    'default' => '',
                ),
            ),
        ));
        
        // 批量 Sideload 端点
        register_rest_route('ai-writer/v1', '/sideload-batch', array(
            'methods' => 'POST',
            'callback' => array($this, 'handle_sideload_batch'),
            'permission_callback' => array($this, 'check_upload_permission'),
            'args' => array(
                'images' => array(
                    'required' => true,
                    'type' => 'array',
                    'description' => '图片数组，每个元素包含 image_url, filename, alt_text, title',
                ),
            ),
        ));
    }
    
    /**
     * Ping 端点处理：返回插件状态
     */
    public function handle_ping($request) {
        return new WP_REST_Response(array(
            'ok' => true,
            'plugin' => 'ai-writer-api-whitelist',
            'version' => self::VERSION,
            'features' => array('sideload', 'sideload-batch', 'resize', 'ping'),
            'image_sizes' => array(
                'article' => array('width' => self::AIWRITER_IMAGE_WIDTH, 'height' => self::AIWRITER_IMAGE_HEIGHT),
                'cover' => array('width' => 600, 'height' => 600),
            ),
            'php_version' => PHP_VERSION,
            'wp_version' => get_bloginfo('version'),
        ), 200);
    }
    
    /**
     * 检查上传权限
     */
    public function check_upload_permission($request) {
        // 获取当前用户（通过 Basic Auth 认证）
        $user = wp_get_current_user();
        
        if (!$user || $user->ID === 0) {
            return new WP_Error(
                'rest_not_logged_in',
                '需要认证才能上传图片',
                array('status' => 401)
            );
        }
        
        // 检查用户是否有上传权限
        if (!current_user_can('upload_files')) {
            return new WP_Error(
                'rest_cannot_upload',
                '当前用户没有上传文件的权限',
                array('status' => 403)
            );
        }
        
        return true;
    }
    
    /**
     * Sideload 端点处理：下载图片并存入媒体库
     */
    public function handle_sideload($request) {
        $image_url = $request->get_param('image_url');
        $filename = $request->get_param('filename');
        $alt_text = $request->get_param('alt_text');
        $title = $request->get_param('title');
        $resize = $request->get_param('resize');
        
        $result = $this->sideload_image($image_url, $filename, $alt_text, $title, $resize);
        
        if (is_wp_error($result)) {
            return new WP_REST_Response(array(
                'ok' => false,
                'error' => $result->get_error_message(),
            ), 500);
        }
        
        return new WP_REST_Response(array(
            'ok' => true,
            'id' => $result['id'],
            'url' => $result['url'],
            'source_url' => $result['url'],  // 兼容标准 media 端点返回格式
            'filename' => $result['filename'],
            'resized' => $result['resized'],
            'size' => $result['size'],
        ), 201);
    }
    
    /**
     * 批量 Sideload 端点处理
     */
    public function handle_sideload_batch($request) {
        $images = $request->get_param('images');
        
        if (!is_array($images) || empty($images)) {
            return new WP_REST_Response(array(
                'ok' => false,
                'error' => '需要提供图片数组',
            ), 400);
        }
        
        if (count($images) > 10) {
            return new WP_REST_Response(array(
                'ok' => false,
                'error' => '单次最多处理 10 张图片',
            ), 400);
        }
        
        $results = array();
        $success_count = 0;
        
        foreach ($images as $index => $img) {
            $image_url = isset($img['image_url']) ? $img['image_url'] : '';
            $filename = isset($img['filename']) ? $img['filename'] : '';
            $alt_text = isset($img['alt_text']) ? $img['alt_text'] : '';
            $title = isset($img['title']) ? $img['title'] : '';
            
            if (empty($image_url)) {
                $results[] = array(
                    'ok' => false,
                    'error' => '缺少 image_url',
                    'index' => $index,
                );
                continue;
            }
            
            $result = $this->sideload_image($image_url, $filename, $alt_text, $title);
            
            if (is_wp_error($result)) {
                $results[] = array(
                    'ok' => false,
                    'error' => $result->get_error_message(),
                    'index' => $index,
                );
            } else {
                $success_count++;
                $results[] = array(
                    'ok' => true,
                    'id' => $result['id'],
                    'url' => $result['url'],
                    'source_url' => $result['url'],
                    'filename' => $result['filename'],
                    'index' => $index,
                );
            }
        }
        
        return new WP_REST_Response(array(
            'ok' => $success_count > 0,
            'total' => count($images),
            'success_count' => $success_count,
            'results' => $results,
        ), 200);
    }
    
    /**
     * 核心函数：下载图片并存入媒体库（支持 resize）
     */
    private function sideload_image($image_url, $filename = '', $alt_text = '', $title = '', $resize = '') {
        // 加载必要的 WordPress 函数
        require_once(ABSPATH . 'wp-admin/includes/media.php');
        require_once(ABSPATH . 'wp-admin/includes/file.php');
        require_once(ABSPATH . 'wp-admin/includes/image.php');
        
        // 记录日志
        if (get_option('ai_writer_enable_logging', false)) {
            $this->log_request('sideload: ' . $image_url . ' resize: ' . $resize);
        }
        
        // 下载图片到临时文件
        $tmp_file = download_url($image_url, 30); // 30秒超时
        
        if (is_wp_error($tmp_file)) {
            return new WP_Error(
                'download_failed',
                '下载图片失败: ' . $tmp_file->get_error_message()
            );
        }
        
        // 解析 resize 参数
        $target_width = 0;
        $target_height = 0;
        $resized = false;
        
        if (!empty($resize)) {
            if ($resize === 'article') {
                $target_width = self::AIWRITER_IMAGE_WIDTH;  // 600
                $target_height = self::AIWRITER_IMAGE_HEIGHT; // 400
            } elseif ($resize === 'cover') {
                $target_width = 600;
                $target_height = 600;
            } elseif (preg_match('/^(\d+)x(\d+)$/i', $resize, $matches)) {
                $target_width = intval($matches[1]);
                $target_height = intval($matches[2]);
            }
        }
        
        // 如果需要 resize，使用 WordPress 图像编辑器处理
        if ($target_width > 0 && $target_height > 0) {
            $resize_result = $this->resize_image_file($tmp_file, $target_width, $target_height);
            if (!is_wp_error($resize_result)) {
                $tmp_file = $resize_result;
                $resized = true;
            }
        }
        
        // 确定文件名
        if (empty($filename)) {
            // 从 URL 提取文件名
            $url_path = parse_url($image_url, PHP_URL_PATH);
            $filename = basename($url_path);
            
            // 如果没有扩展名，根据 MIME 类型添加
            if (!preg_match('/\.(jpg|jpeg|png|gif|webp)$/i', $filename)) {
                $mime_type = mime_content_type($tmp_file);
                $ext = $this->get_extension_from_mime($mime_type);
                $filename = 'ai-writer-' . time() . '-' . wp_rand(1000, 9999) . '.' . $ext;
            }
        }
        
        // 确保文件名安全
        $filename = sanitize_file_name($filename);
        
        // 准备文件数组
        $file_array = array(
            'name' => $filename,
            'tmp_name' => $tmp_file,
        );
        
        // 使用 media_handle_sideload 将文件添加到媒体库
        $attachment_id = media_handle_sideload($file_array, 0, $title);
        
        // 清理临时文件（如果还存在）
        if (file_exists($tmp_file)) {
            @unlink($tmp_file);
        }
        
        if (is_wp_error($attachment_id)) {
            return new WP_Error(
                'sideload_failed',
                '保存到媒体库失败: ' . $attachment_id->get_error_message()
            );
        }
        
        // 设置 alt 文本
        if (!empty($alt_text)) {
            update_post_meta($attachment_id, '_wp_attachment_image_alt', sanitize_text_field($alt_text));
        }
        
        // 设置标题
        if (!empty($title)) {
            wp_update_post(array(
                'ID' => $attachment_id,
                'post_title' => sanitize_text_field($title),
            ));
        }
        
        // 获取附件 URL 和尺寸信息
        $attachment_url = wp_get_attachment_url($attachment_id);
        $metadata = wp_get_attachment_metadata($attachment_id);
        $size = array(
            'width' => isset($metadata['width']) ? $metadata['width'] : 0,
            'height' => isset($metadata['height']) ? $metadata['height'] : 0,
        );
        
        return array(
            'id' => $attachment_id,
            'url' => $attachment_url,
            'filename' => $filename,
            'resized' => $resized,
            'size' => $size,
        );
    }
    
    /**
     * 使用 WordPress 图像编辑器调整图片尺寸（居中裁剪）
     * @param string $file_path 临时文件路径
     * @param int $target_width 目标宽度
     * @param int $target_height 目标高度
     * @return string|WP_Error 新文件路径或错误
     */
    private function resize_image_file($file_path, $target_width, $target_height) {
        // 获取图像编辑器实例
        $editor = wp_get_image_editor($file_path);
        
        if (is_wp_error($editor)) {
            return $editor;
        }
        
        // 获取原始尺寸
        $size = $editor->get_size();
        $orig_width = $size['width'];
        $orig_height = $size['height'];
        
        // 计算目标比例和原始比例
        $target_ratio = $target_width / $target_height;
        $orig_ratio = $orig_width / $orig_height;
        
        // 计算裁剪区域（居中裁剪以匹配目标比例）
        if ($orig_ratio > $target_ratio) {
            // 原图更宽，需要裁剪左右
            $crop_height = $orig_height;
            $crop_width = intval($orig_height * $target_ratio);
            $crop_x = intval(($orig_width - $crop_width) / 2);
            $crop_y = 0;
        } else {
            // 原图更高，需要裁剪上下
            $crop_width = $orig_width;
            $crop_height = intval($orig_width / $target_ratio);
            $crop_x = 0;
            $crop_y = intval(($orig_height - $crop_height) / 2);
        }
        
        // 先裁剪到正确比例
        $result = $editor->crop($crop_x, $crop_y, $crop_width, $crop_height);
        if (is_wp_error($result)) {
            return $result;
        }
        
        // 然后缩放到目标尺寸
        $result = $editor->resize($target_width, $target_height, false);
        if (is_wp_error($result)) {
            return $result;
        }
        
        // 设置 JPEG 质量
        $editor->set_quality(92);
        
        // 保存到新的临时文件
        $new_file = $file_path . '-resized.jpg';
        $saved = $editor->save($new_file, 'image/jpeg');
        
        if (is_wp_error($saved)) {
            return $saved;
        }
        
        // 删除原临时文件
        @unlink($file_path);
        
        return $saved['path'];
    }
    
    /**
     * 根据 MIME 类型获取文件扩展名
     */
    private function get_extension_from_mime($mime_type) {
        $mime_to_ext = array(
            'image/jpeg' => 'jpg',
            'image/png' => 'png',
            'image/gif' => 'gif',
            'image/webp' => 'webp',
        );
        
        return isset($mime_to_ext[$mime_type]) ? $mime_to_ext[$mime_type] : 'jpg';
    }
    
    /**
     * 检查是否是 AI Writer 的请求
     */
    private function is_ai_writer_request() {
        // 检查是否是 REST API 请求
        $request_uri = isset($_SERVER['REQUEST_URI']) ? $_SERVER['REQUEST_URI'] : '';
        
        // 检查标准 REST API 路径
        $is_rest = (strpos($request_uri, '/wp-json/') !== false);
        
        // 检查 rest_route 查询参数（绕过 SiteGround WAF 的方式）
        $is_rest_route = isset($_GET['rest_route']) || (strpos($request_uri, 'rest_route=') !== false);
        
        if (!$is_rest && !$is_rest_route) {
            return false;
        }
        
        // 检查 IP 白名单
        $client_ip = $this->get_client_ip();
        $custom_ips = get_option('ai_writer_custom_ips', '');
        $all_ips = $this->whitelisted_ips;
        
        if (!empty($custom_ips)) {
            $custom_ip_array = array_filter(array_map('trim', explode("\n", $custom_ips)));
            $all_ips = array_merge($all_ips, $custom_ip_array);
        }
        
        if (in_array($client_ip, $all_ips)) {
            return true;
        }
        
        // 检查 Authorization 头（应用密码认证）
        $auth_header = $this->get_authorization_header();
        if (!empty($auth_header) && strpos($auth_header, 'Basic ') === 0) {
            // 有有效的 Basic Auth 头，可能是 AI Writer 请求
            return true;
        }
        
        return false;
    }
    
    /**
     * 获取客户端真实 IP
     */
    private function get_client_ip() {
        $ip_keys = array(
            'HTTP_CF_CONNECTING_IP',     // Cloudflare
            'HTTP_X_FORWARDED_FOR',      // 代理
            'HTTP_X_REAL_IP',            // Nginx
            'HTTP_CLIENT_IP',
            'REMOTE_ADDR'
        );
        
        foreach ($ip_keys as $key) {
            if (!empty($_SERVER[$key])) {
                $ip = $_SERVER[$key];
                // 处理多个 IP 的情况（X-Forwarded-For）
                if (strpos($ip, ',') !== false) {
                    $ips = explode(',', $ip);
                    $ip = trim($ips[0]);
                }
                if (filter_var($ip, FILTER_VALIDATE_IP)) {
                    return $ip;
                }
            }
        }
        
        return '';
    }
    
    /**
     * 获取 Authorization 头
     */
    private function get_authorization_header() {
        if (isset($_SERVER['HTTP_AUTHORIZATION'])) {
            return $_SERVER['HTTP_AUTHORIZATION'];
        }
        if (isset($_SERVER['REDIRECT_HTTP_AUTHORIZATION'])) {
            return $_SERVER['REDIRECT_HTTP_AUTHORIZATION'];
        }
        if (function_exists('apache_request_headers')) {
            $headers = apache_request_headers();
            if (isset($headers['Authorization'])) {
                return $headers['Authorization'];
            }
            if (isset($headers['authorization'])) {
                return $headers['authorization'];
            }
        }
        return '';
    }
    
    /**
     * 绕过安全插件检查
     */
    public function maybe_bypass_security() {
        if (!$this->is_ai_writer_request()) {
            return;
        }
        
        // 记录日志（可选）
        if (get_option('ai_writer_enable_logging', false)) {
            $this->log_request();
        }
        
        // 设置标记，告诉其他安全插件跳过检查
        if (!defined('AI_WRITER_WHITELISTED')) {
            define('AI_WRITER_WHITELISTED', true);
        }
        
        // 禁用 SG Security 的 bot 检测
        if (!defined('SG_SECURITY_BYPASS')) {
            define('SG_SECURITY_BYPASS', true);
        }
        
        // 尝试禁用其他常见安全插件的检测
        // Wordfence
        if (!defined('WORDFENCE_DISABLE_LIVE_TRAFFIC')) {
            define('WORDFENCE_DISABLE_LIVE_TRAFFIC', true);
        }
        
        // 设置 cookie 绕过验证码
        if (!isset($_COOKIE['sg-captcha-bypass'])) {
            setcookie('sg-captcha-bypass', '1', time() + 3600, '/');
        }
    }
    
    /**
     * 禁用 API 请求的安全检查
     */
    public function disable_security_for_api() {
        if (!$this->is_ai_writer_request()) {
            return;
        }
        
        // 移除可能干扰的 action
        // SG Security
        remove_all_actions('sg_security_before_request');
        
        // 确保 REST API 可用
        add_filter('rest_enabled', '__return_true');
        add_filter('rest_jsonp_enabled', '__return_true');
        
        // 允许媒体上传
        add_filter('upload_mimes', array($this, 'allow_all_mimes'), 100);
    }
    
    /**
     * 允许 AI Writer 请求通过 REST 认证
     */
    public function allow_ai_writer_requests($errors) {
        if ($this->is_ai_writer_request()) {
            // 清除之前的错误
            return null;
        }
        return $errors;
    }
    
    /**
     * 允许更多 MIME 类型
     */
    public function allow_all_mimes($mimes) {
        $mimes['jpg'] = 'image/jpeg';
        $mimes['jpeg'] = 'image/jpeg';
        $mimes['png'] = 'image/png';
        $mimes['gif'] = 'image/gif';
        $mimes['webp'] = 'image/webp';
        return $mimes;
    }
    
    /**
     * 记录请求日志
     */
    private function log_request($extra = '') {
        $log_file = WP_CONTENT_DIR . '/ai-writer-api.log';
        $log_entry = sprintf(
            "[%s] IP: %s | URI: %s | Method: %s %s\n",
            date('Y-m-d H:i:s'),
            $this->get_client_ip(),
            isset($_SERVER['REQUEST_URI']) ? $_SERVER['REQUEST_URI'] : '',
            isset($_SERVER['REQUEST_METHOD']) ? $_SERVER['REQUEST_METHOD'] : '',
            $extra ? '| ' . $extra : ''
        );
        file_put_contents($log_file, $log_entry, FILE_APPEND | LOCK_EX);
    }
    
    /**
     * 添加管理菜单
     */
    public function add_admin_menu() {
        add_options_page(
            'AI Writer API 白名单',
            'AI Writer API',
            'manage_options',
            'ai-writer-api-whitelist',
            array($this, 'render_admin_page')
        );
    }
    
    /**
     * 注册设置
     */
    public function register_settings() {
        register_setting('ai_writer_api_settings', 'ai_writer_custom_ips');
        register_setting('ai_writer_api_settings', 'ai_writer_enable_logging');
    }
    
    /**
     * 渲染管理页面
     */
    public function render_admin_page() {
        if (!current_user_can('manage_options')) {
            return;
        }
        
        // 保存设置
        if (isset($_POST['submit']) && check_admin_referer('ai_writer_api_settings_nonce')) {
            update_option('ai_writer_custom_ips', sanitize_textarea_field($_POST['ai_writer_custom_ips'] ?? ''));
            update_option('ai_writer_enable_logging', isset($_POST['ai_writer_enable_logging']) ? 1 : 0);
            echo '<div class="notice notice-success"><p>设置已保存！</p></div>';
        }
        
        $custom_ips = get_option('ai_writer_custom_ips', '');
        $enable_logging = get_option('ai_writer_enable_logging', false);
        $client_ip = $this->get_client_ip();
        
        // 测试 Sideload 端点
        $sideload_test_url = rest_url('ai-writer/v1/ping');
        
        ?>
        <div class="wrap">
            <h1>AI Writer API 白名单设置 v<?php echo self::VERSION; ?></h1>
            
            <div class="card" style="max-width: 800px; padding: 20px; margin-bottom: 20px;">
                <h2>当前状态</h2>
                <p><strong>您的 IP 地址：</strong> <?php echo esc_html($client_ip); ?></p>
                <p><strong>Ping 端点：</strong> <code><?php echo esc_html($sideload_test_url); ?></code></p>
                <p><strong>插件版本：</strong> <?php echo self::VERSION; ?></p>
                <p><strong>支持的功能：</strong> sideload, sideload-batch, ping</p>
                
                <h3>默认白名单 IP</h3>
                <ul>
                    <?php foreach ($this->whitelisted_ips as $ip): ?>
                        <li><code><?php echo esc_html($ip); ?></code> (AI Writer Zeabur 服务器)</li>
                    <?php endforeach; ?>
                </ul>
            </div>
            
            <form method="post" action="">
                <?php wp_nonce_field('ai_writer_api_settings_nonce'); ?>
                
                <table class="form-table">
                    <tr>
                        <th scope="row">
                            <label for="ai_writer_custom_ips">自定义白名单 IP</label>
                        </th>
                        <td>
                            <textarea name="ai_writer_custom_ips" id="ai_writer_custom_ips" 
                                      rows="5" cols="50" class="large-text code"
                                      placeholder="每行一个 IP 地址"><?php echo esc_textarea($custom_ips); ?></textarea>
                            <p class="description">
                                如果 AI Writer 使用了不同的服务器 IP，请在这里添加。每行一个 IP 地址。
                            </p>
                        </td>
                    </tr>
                    <tr>
                        <th scope="row">启用日志</th>
                        <td>
                            <label>
                                <input type="checkbox" name="ai_writer_enable_logging" value="1" 
                                       <?php checked($enable_logging, 1); ?>>
                                记录 AI Writer API 请求日志
                            </label>
                            <p class="description">
                                日志文件位置：<code><?php echo esc_html(WP_CONTENT_DIR . '/ai-writer-api.log'); ?></code>
                            </p>
                        </td>
                    </tr>
                </table>
                
                <?php submit_button('保存设置'); ?>
            </form>
            
            <div class="card" style="max-width: 800px; padding: 20px; margin-top: 20px;">
                <h2>使用说明</h2>
                <h3>新功能：Sideload 模式（v2.0）</h3>
                <p>此版本支持 <strong>Sideload 模式</strong>：AI Writer 只发送图片 URL，WordPress 服务器自行下载图片并存入媒体库。这种方式：</p>
                <ul>
                    <li>✅ 完全绕过 SiteGround 等主机的 WAF 限制</li>
                    <li>✅ 请求体积更小，更稳定</li>
                    <li>✅ 支持批量处理</li>
                </ul>
                
                <h3>API 端点</h3>
                <ul>
                    <li><code>GET /?rest_route=/ai-writer/v1/ping</code> - 检测插件是否安装</li>
                    <li><code>POST /?rest_route=/ai-writer/v1/sideload</code> - 下载单张图片</li>
                    <li><code>POST /?rest_route=/ai-writer/v1/sideload-batch</code> - 批量下载图片</li>
                </ul>
                
                <h3>故障排除</h3>
                <p>如果安装插件后仍然无法正常使用：</p>
                <ol>
                    <li>确保插件已激活</li>
                    <li>检查 WordPress 用户是否有"上传文件"权限</li>
                    <li>启用日志功能，查看详细错误信息</li>
                    <li>确保服务器可以访问外部 URL（用于下载图片）</li>
                </ol>
            </div>
        </div>
        <?php
    }
}

// 初始化插件
new AI_Writer_API_Whitelist();

// 激活钩子
register_activation_hook(__FILE__, function() {
    // 刷新重写规则
    flush_rewrite_rules();
});

// 停用钩子
register_deactivation_hook(__FILE__, function() {
    // 清理设置
    delete_option('ai_writer_custom_ips');
    delete_option('ai_writer_enable_logging');
    flush_rewrite_rules();
});
